/*
 *
 * wsstdlib.c
 *
 * Author: Markku Rossi <mtr@iki.fi>
 *
 * Copyright (c) 1999-2000 WAPIT OY LTD.
 *       All rights reserved.
 *
 * Standard libraries.
 *
 */

#include <expat.h>
#include "wsint.h"

/* TODO: the function registry could have argument type specifier
 * strings.  These could be used to generate extra warnings when
 * functions are called with wrong arguments.  However, this might not
 * be fully conformable to the WMLScript specification.  I think that
 * the interpreter might do an automatic type conversion so the
 * warnings are less useful.  But, these warnings could be enabled in
 * the `-Wpedantic' mode. */

/********************* Types and definitions ****************************/

/* Calculate the number of function registries in the array `f'. */
#define NF(f) (sizeof(f) / sizeof(f[0]))

#define MAX_LIBS  256
#define MAX_FUNCS 256

/* Information about a standard library function. */
struct WsStdLibFuncRegRec
{
    char *name;

    /* The exact number of arguments. */
    int num_args;
    WsUInt8 function_id;
};

typedef struct WsStdLibFuncRegRec WsStdLibFuncReg;

/* Information about a standard library. */
struct WsStdLibRegRec
{
    char *name;

    WsUInt16 library_id;

    /* The number of functions in this library. */
    WsUInt8 num_functions;

    /* The functions are given in their index order. */
    WsStdLibFuncReg *functions;
};

typedef struct WsStdLibRegRec WsStdLibReg;

/********************* Static variables *********************************/

static WsStdLibFuncReg lib_lang_functions[] =
    {
        {"abs", 1, 0},
        {"min", 2, 1},
        {"max", 2, 2},
        {"parseInt", 1, 3},
        {"parseFloat", 1, 4},
        {"isInt", 1, 5},
        {"isFloat", 1, 6},
        {"maxInt", 0, 7},
        {"minInt", 0, 8},
        {"float", 0, 9},
        {"exit", 1, 10},
        {"abort", 1, 11},
        {"random", 1, 12},
        {"seed", 1, 13},
        {"characterSet", 0, 14},
    };

static WsStdLibFuncReg lib_float_functions[] =
    {
        {"int", 1, 0},
        {"floor", 1, 1},
        {"ceil", 1, 2},
        {"pow", 2, 3},
        {"round", 1, 4},
        {"sqrt", 1, 5},
        {"maxFloat", 0, 6},
        {"minFloat", 0, 7},
    };

static WsStdLibFuncReg lib_string_functions[] =
    {
        {"length", 1, 0},
        {"isEmpty", 1, 1},
        {"charAt", 2, 2},
        {"subString", 3, 3},
        {"find", 2, 4},
        {"replace", 3, 5},
        {"elements", 2, 6},
        {"elementAt", 3, 7},
        {"removeAt", 3, 8},
        {"replaceAt", 4, 9},
        {"insertAt", 4, 10},
        {"squeeze", 1, 11},
        {"trim", 1, 12},
        {"compare", 2, 13},
        {"toString", 1, 14},
        {"format", 2, 15},
        {"replaceChars", 3, 16},
    };

static WsStdLibFuncReg lib_env_functions[MAX_FUNCS];

static WsStdLibReg libraries[MAX_LIBS] =
    {
        {"Lang", 0, NF(lib_lang_functions), lib_lang_functions},
        {"Float", 1, NF(lib_float_functions), lib_float_functions},
        {"String", 2, NF(lib_string_functions), lib_string_functions},
        {NULL, 0, 0, NULL}
    };


static enum env_state { env_initial, env_final, env_env, env_lib } state;

static int lcount = 3; // this includes 3 standard libraries
static int fcount = 0;


/********************* Global functions *********************************/

WsBool ws_stdlib_function(const char *library, const char *function,
                          WsUInt16 *lindex_return, WsUInt8 *findex_return,
                          WsUInt8 *num_args_return, WsBool *lindex_found_return,
                          WsBool *findex_found_return)
{
    WsUInt16 l;

    *lindex_found_return = WS_FALSE;
    *findex_found_return = WS_FALSE;

    for (l = 0; libraries[l].name != NULL; l++) {
        if (strcmp(libraries[l].name, library) == 0) {
            WsUInt8 f;

            *lindex_return = libraries[l].library_id;
            *lindex_found_return = WS_TRUE;

            for (f = 0; f < libraries[l].num_functions; f++) {
                if (strcmp(libraries[l].functions[f].name, function) == 0) {
                    *findex_return = libraries[l].functions[f].function_id;
                    *findex_found_return = WS_TRUE;

                    *num_args_return = libraries[l].functions[f].num_args;

                    return WS_TRUE;
                }
        }
        }
    }

    return WS_FALSE;
}


WsBool ws_stdlib_function_name(WsUInt16 lindex, WsUInt8 findex,
                               const char **library_return,
                               const char **function_return)
{
    WsUInt16 l;
    WsUInt8 f;

    *library_return = NULL;
    *function_return = NULL;

    for (l = 0; libraries[l].name != NULL; l++)
        if (libraries[l].library_id == lindex)
            for (f = 0; f < libraries[l].num_functions; f++) {
                if (libraries[l].functions[f].function_id == findex) {
                    *library_return = libraries[l].name;
                    *function_return = libraries[l].functions[f].name;
                    return WS_TRUE;
                }
            }

    return WS_FALSE;
}

WsBool ws_stdlib_parse_env(const char *fname)
{
    static void handle_start(void *context, const XML_Char *name, const XML_Char **attrs);
    static void handle_end(void *context, const XML_Char *name);

    FILE *fd = fopen(fname, "rt");
    if(fd)
    {
        char buf[256]; // TODO: hardcoded value!!!
        enum XML_Status status = XML_STATUS_OK;
        XML_Parser parser = XML_ParserCreate(NULL);

        XML_SetElementHandler(parser,
            (XML_StartElementHandler)handle_start,
            (XML_EndElementHandler)handle_end);
//      XML_SetUserData(parser, &context);

        state = env_initial;

        // parse content
        while(status == XML_STATUS_OK && fgets(buf, sizeof(buf), fd))
            status = XML_Parse(parser, buf, strlen(buf), 0);

        // cleanup
        XML_Parse(parser, buf, 0, 1);
        XML_ParserFree(parser);

        fclose(fd);
    }

    return WS_TRUE;
}

/********************* Helper functions *********************************/

static void env_error(const char *message, ...)
{
    va_list args;
    va_start(args, message);
    
    vfprintf(stderr, message, args);
    exit(-1);
}

static char *stralloc(const char *str)
{
    char *ptr = malloc(strlen(str) + 1);
    strcpy(ptr, str);
    return ptr;
}

static void env_library_begin(const char *name, WsUInt16 library_id)
{
    if(lcount >= MAX_LIBS)
        env_error("Maximum number of libraries reached,\n");

    libraries[lcount].name = stralloc(name);
    libraries[lcount].library_id = library_id;

    fcount = 0;
}

static void env_library_end()
{
    WsStdLibFuncReg *functions = malloc(sizeof(WsStdLibFuncReg) * fcount);

    memmove(functions, lib_env_functions, sizeof(WsStdLibFuncReg) * fcount);
    libraries[lcount].num_functions = fcount;
    libraries[lcount].functions =  functions;

    lcount++;
}


static void env_function(const char *name, int argc, WsUInt8 function_id)
{
    if(fcount >= MAX_FUNCS)
        env_error("Maximum number of functions in library reached,\n");

    lib_env_functions[fcount].name = stralloc(name);
    lib_env_functions[fcount].num_args = argc;
    lib_env_functions[fcount].function_id = function_id;

    fcount++;
}


static const XML_Char *get_attr(const XML_Char **attrs, const XML_Char *name)
{
    while(*attrs)
    {
        if(strcmp(*attrs++, name) == 0)
            return *attrs;
        attrs++;
    }

    env_error("Missing `%s' attribute.\n", name);
    return NULL;
}

static void handle_start(void *context, const XML_Char *element, const XML_Char **attrs)
{
    const XML_Char *name, *argc, *index;

    if(strcmp(element, "function") == 0)
    {
        if(state != env_lib)
            env_error("Unexpected `%s' element.\n", element);

        name  = get_attr(attrs, "name");
        argc  = get_attr(attrs, "argc");
        index = get_attr(attrs, "index");

        env_function(name, atoi(argc), atoi(index));
        return;
    }

    if(strcmp(element, "library") == 0)
    {
        if(state != env_env)
            env_error("Unexpected `%s' element.\n", element);

        state = env_lib;
        name  = get_attr(attrs, "name");
        index = get_attr(attrs, "index");

        env_library_begin(name, atoi(index));
        return;
    }

    if(strcmp(element, "environment") == 0)
    {
        if(state != env_initial)
            env_error("Unexpected `%s' element.\n", element);

        state = env_env;
        return;
    }

    env_error("Unexpected element `%s' in environment definition file.\n", element);
    exit(-1);
}

static void handle_end(void *context, const XML_Char *name)
{
    if(strcmp(name, "library") == 0)
    {
        env_library_end();
        state = env_env;
    }

    if(strcmp(name, "environment") == 0)
        state = env_final;
}
